library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

library unisim;
use unisim.VComponents.all;

entity multi_boot is
    generic(
				C_WARM_BOOT_ADDR    : std_logic_vector  := x"00200000"
           );

    Port  ( clock_in : in  std_logic;
	         re_configure : in std_logic
	       );
end multi_boot;


architecture Behavioral of multi_boot is
type controller_state is (S_RESET,S_WRT,S_CE,S_ICAP,S_ERROR);
signal current_state, next_state : controller_state;


signal CE:  std_logic :='1';
signal CLK:  std_logic;
signal I:  std_logic_vector(31 downto 0):= "ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ";
signal WRITE: std_logic := '1';
signal RESET: std_logic :='1';
signal COUNT : integer range 0 to 15 := 0;
signal DONE_WRT : std_logic := '0';
signal DONE_CE : std_logic := '0';
signal DONE_ICAP : std_logic := '0';	



begin
ICAP_VIRTEX5_inst : ICAP_VIRTEX5
generic map (ICAP_WIDTH => "X32") -- "X8" or "X32
port map (
           CE    => CE,        -- Clock enable input
           CLK   => CLK,       -- Clock input
           I     => I,         -- 32-bit data input
           WRITE => WRITE      -- Write input
         );



SYNC_PROC: process(clock_in)
begin
	if(falling_edge(clock_in)) then
		if (RESET = '1') then
			current_state <= S_RESET;
		else
			current_state <= next_state;
		end if;
	end if;
		
end process SYNC_PROC;



MAIN_PROC: process(current_state,re_configure,DONE_WRT,DONE_CE,DONE_ICAP)
begin
	case current_state is
		when S_RESET =>
			if(re_configure = '1') then
				next_state <= S_WRT;
			else
				next_state <= S_RESET;
			end if;
		when S_WRT =>
			if(DONE_WRT = '1') then
				next_state <= S_CE;
			else
				next_state <= S_WRT;
			end if;
		when S_CE =>
			if(DONE_CE = '1') then
				next_state <=  S_ICAP;
			else
				next_state <= S_CE;
			end if;
		when S_ICAP =>
			if(DONE_ICAP = '1') then
				next_state <=  S_ERROR;
			else
				next_state <= S_ICAP;
			end if;
		when S_ERROR =>
				next_state <= S_ERROR;
		when others=>
	end case;
end process MAIN_PROC;



PROC: process(clock_in)
begin
if(falling_edge(clock_in)) then
	case next_state is
	when S_RESET =>
			DONE_WRT <= '0';
			DONE_CE <= '0';
			DONE_ICAP <= '0';
			WRITE <= '1';
			CE <= '1';
			RESET <= '0';
	when S_WRT =>
			DONE_WRT <= '1';
			WRITE <= '0';
	when S_CE =>
			DONE_CE<= '1';
			CE <= '0';
	when S_ICAP =>
		case COUNT is
			when 0 =>  	--0
				I <= x"FFFFFFFF";		    -- Dummy Word
				COUNT <= COUNT + 1;
			when 1 =>		--1
				I <= x"5599AA66";		    -- Sync Word
				COUNT <= COUNT + 1;
			when 2 =>		--2
				I <= x"04000000";		    -- type 1 NO OP
				COUNT <= COUNT + 1;
			when 3 =>		--7
				I <= x"0C400080";		    -- Type 1 Write 1 Words to WBSTAR
				COUNT <= COUNT + 1;
			when 4 =>		--8
				I <= C_WARM_BOOT_ADDR;   -- Warm Boot Start Address (Load the Desired Address
				
				COUNT <= COUNT + 1;
			when 5 =>		--9
				I <= x"0C000180";        -- Type 1 Write 1 Words to CMD
				COUNT <= COUNT + 1;
			when 6 =>		--10
				I <= x"000000F0"; 	    -- IPROG Command
				COUNT <= COUNT + 1;
			when 7 =>		--11
				I <= x"04000000";		    -- Type 1 NO OP
				COUNT <= COUNT + 1;
				CE <= '1';
				DONE_ICAP <= '1';
			when others=>
				COUNT <= COUNT + 1;
				I <= x"AAAAAAAA";				
		end case;
		
	when S_ERROR =>
		WRITE <= '0';
		--- ERROR
	when others =>
		RESET <= '1';	
	end case;
		
end if;
end process PROC;

				

CLK <= clock_in;
			 
end Behavioral;

