/* Name: main.c
 * Project: PowerSwitch based on AVR USB driver
 * Author: Christian Starkjohann
 * Creation Date: 2005-01-16
 * Tabsize: 4
 * Copyright: (c) 2005 by OBJECTIVE DEVELOPMENT Software GmbH
 * License: GNU GPL v2 (see License.txt) or proprietary (CommercialLicense.txt)
 * This Revision: $Id: main.c 523 2008-02-15 09:46:40Z cs $
 */

#include <avr/io.h>
#define F_CPU 1000000
#include <util/delay.h>
#include <string.h>
#include "types.h"

#define LED_C0_PIN 4
#define LED_C1_PIN 1
#define LED_C2_PIN 2
#define LED_C3_PIN 3    

#define LED_C_PORT PORTC
#define LED_C_DDR  DDRC
#define LED_A_PORT PORTD
#define LED_A_DDR  DDRD

void led_init(void)
{
  sbi(LED_C_PORT,LED_C0_PIN);
  sbi(LED_C_PORT,LED_C1_PIN);
  sbi(LED_C_PORT,LED_C2_PIN);
  sbi(LED_C_PORT,LED_C3_PIN);

  sbi(LED_C_DDR,LED_C0_PIN);
  sbi(LED_C_DDR,LED_C1_PIN);
  sbi(LED_C_DDR,LED_C2_PIN);
  sbi(LED_C_DDR,LED_C3_PIN);

  LED_A_DDR = 0xFF;
}

#define MAX_DIGIT 4

u08 digit=0x00;
u08 digits[MAX_DIGIT]={0x00};

void set_port_c(u08 d)
{
  LED_C_PORT&=0xE1;
  LED_C_PORT|=~(1<<(d+1));
}

u08  adc_chn(u08 chn)
{
// Wybór kanału wejścia - PC3 (ADC3)
ADMUX &= 0xF0;
ADMUX |= chn;
ADCSRA |= _BV(ADSC);
while( ADCSRA & _BV(ADSC)) {}; 

return ADCH>>1;
}


void adc_init()
{
ADMUX |= _BV(REFS0);
ADMUX |= _BV(REFS1);
// Wybranie sposobu zapisu wyniku z wyrównaniem do lewej (osiem starszych bitów wyniku w rejestrze ADCH)
//ADMUX |= _BV(ADLAR);
// Zezwolenie na konwersję
ADCSRA |= _BV(ADEN);
// Wybranie częstotliwości dla taktowania przetwornika
ADCSRA |= _BV(ADPS0);
ADCSRA |= _BV(ADPS1);
ADCSRA |= _BV(ADPS2);



}




#define SEGA  (1<<0)
#define SEGB  (1<<2)
#define SEGC  (1<<6)
#define SEGD  (1<<4)
#define SEGE  (1<<3)
#define SEGG  (1<<7)
#define SEGF  (1<<1)
#define SEGDP (1<<5)

#define DISP_OFF 0xFF;

#define IN_PORT PORTB
#define IN_DDR  DDRB
#define IN_PIN  PINB

#define IN0 (1<<0)
#define IN1 (1<<1)
#define IN2 (1<<2)
#define IN3 (1<<3)
#define IN4 (1<<4)
#define IN5 (1<<5)
#define IN6 (1<<6)
#define IN7 (1<<7)


u08 bin_to_7seg(u08 i)
{
 u08 r=0;
 switch(i)
 { 
   case 0x0 : r = SEGA | SEGB | SEGC | SEGD | SEGE | SEGF ; break;
   case 0x1 : r = SEGB | SEGC ; break; 
   case 0x2 : r = SEGA | SEGB | SEGD | SEGE | SEGG ; break; 
   case 0x3 : r = SEGA | SEGB | SEGC | SEGD | SEGG ; break; 
   case 0x4 : r = SEGB | SEGC | SEGF | SEGG ; break; 
   case 0x5 : r = SEGA | SEGC | SEGD | SEGG | SEGF ; break; 
   case 0x6 : r = SEGA | SEGF | SEGC | SEGD | SEGE | SEGG ; break; 
   case 0x7 : r = SEGA | SEGB | SEGC ; break; 
   case 0x8 : r = SEGA | SEGB | SEGC | SEGD | SEGE | SEGF | SEGG ; break; 
   case 0x9 : r = SEGA | SEGB | SEGC | SEGD | SEGG | SEGF ; break; 
   case 0xA : r = SEGA | SEGB | SEGC | SEGE | SEGF | SEGG; break;
   case 0xB : r = SEGC | SEGD | SEGE | SEGF | SEGG ; break; 
   case 0xC : r = SEGA | SEGD | SEGE | SEGF ; break; 
   case 0xD : r = SEGB | SEGC | SEGD | SEGE | SEGG ; break; 
   case 0xE : r = SEGA | SEGD | SEGE | SEGF | SEGG; break; 
   case 0xF : r = SEGA | SEGE | SEGF | SEGG; break; 
   default  : r = 0xFF;
 }
 return ~r;
}

void dec_dips(u08 val, u08 * digits)
{
    if (val<10)
    {
      digits[0] =  bin_to_7seg(val&0xF);
      digits[1]=DISP_OFF;
      digits[2]=DISP_OFF;
      digits[3]=DISP_OFF;
    } else if (val<100)
    {
      u08 j= val%10;
      u08 d= val/10;
      digits[0] =  bin_to_7seg(j);
      digits[1]= bin_to_7seg(d);
      digits[2]=DISP_OFF;
      digits[3]=DISP_OFF;
    } else
    {
      u08 j= val%10;
      u08 d= (val/10)%10;
      u08 s= val/100;

      digits[0]=bin_to_7seg(j);
      digits[1]=bin_to_7seg(d);
      digits[2]=bin_to_7seg(s);
      digits[3]=DISP_OFF;
    } 
}

void hex_dips(u08 val, u08 * digits)
{
    digits[0] = bin_to_7seg(val&0xF);
    digits[1] = bin_to_7seg((val>>4)&0xF);
    digits[2]=DISP_OFF;
    digits[3]=DISP_OFF;
}




u08 gray2bin(u08 v)
{
  u08 V=v;
  u08 M=1<<7;
  while(M>>=1) 
     V^=M&(V>>1);
  return V;
} 


int main(void)
{
 led_init();
 adc_init();
 u08 d;
 digits[0]=bin_to_7seg(0x4);
 digits[1]=DISP_OFF;
 digits[2]=DISP_OFF;
 digits[3]=DISP_OFF;
 IN_DDR=0x00;
 IN_PORT=0xFF   ;
 sbi(PORTC,5);

 while(1)
 {

   for(d=0;d<4;d++)
   {
     set_port_c(d);
     LED_A_PORT=digits[d];//~SEGC;
   _delay_ms(1);
   }


   u08 mode=7;

   if(PINC & (1<<5)) cbi(mode,0);
   if (adc_chn(6))   cbi(mode,1);
   if (adc_chn(7))   cbi(mode,2);

   u08 inputs=~IN_PIN;
   u08 r=0x00;
   
   //zwykly 7 seg
   if (mode ==0 || mode == 4)
   {
    if( inputs & IN0) r |= SEGA; 
    if( inputs & IN1) r |= SEGB; 
    if( inputs & IN2) r |= SEGC; 
    if( inputs & IN3) r |= SEGD; 
    if( inputs & IN4) r |= SEGE; 
    if( inputs & IN5) r |= SEGF; 
    if( inputs & IN6) r |= SEGG; 
    if( inputs & IN7) r |= SEGDP; 
    digits[0] = ~r;
    digits[1]=DISP_OFF;
    digits[2]=DISP_OFF;
    digits[3]=DISP_OFF;
   }
   // bin 2 decymalnie 
   else if (mode ==1 )
   {
     dec_dips(inputs, digits);
   }
   // bin 2 hex
   else if (mode ==5 )
   {
     hex_dips(inputs, digits);
   } 
   // gray 2 decymalnie 
   else if (mode ==2 )
   {  
     u08 g=gray2bin(inputs);
     dec_dips(g, digits);
   }
   // gray 2 hex
   else if (mode ==6 )
   {
     u08 g=gray2bin(inputs);
     hex_dips(g, digits);
   } 
   else
   {
    digits[0] = ~SEGG;
    digits[1]=~SEGG;
    digits[2]=~SEGG;
    digits[3]=~SEGG;
   }
 }
}
